/**
 * Popup Script - Extension UI Logic
 */

// Check authentication on load
document.addEventListener('DOMContentLoaded', async () => {
    const { apiToken } = await chrome.storage.sync.get(['apiToken']);

    if (apiToken) {
        await checkAuth(apiToken);
    } else {
        showLoginView();
    }
});

// Login with Google
document.getElementById('login-google').addEventListener('click', () => {
    // Send message to background script to handle OAuth
    chrome.runtime.sendMessage({
        action: 'startOAuth',
        provider: 'google'
    });
    window.close();
});

// Login with Facebook
document.getElementById('login-facebook').addEventListener('click', () => {
    // Send message to background script to handle OAuth
    chrome.runtime.sendMessage({
        action: 'startOAuth',
        provider: 'facebook'
    });
    window.close();
});

// Save API token manually
document.getElementById('save-token').addEventListener('click', async () => {
    const token = document.getElementById('token-input').value.trim();

    if (!token) {
        showError('Please enter a valid API token');
        return;
    }

    await chrome.storage.sync.set({ apiToken: token });
    await checkAuth(token);
});

// Logout
document.getElementById('logout').addEventListener('click', async () => {
    // Revoke token on server
    const { apiToken } = await chrome.storage.sync.get(['apiToken']);

    if (apiToken) {
        try {
            await fetch('https://journament.com/api/auth/revoke', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${apiToken}`,
                    'Accept': 'application/json'
                }
            });
        } catch (error) {
            // Token revocation failed - continue with logout anyway
        }
    }

    // Clear storage
    await chrome.storage.sync.remove(['apiToken']);
    await chrome.storage.local.remove(['importHistory']);

    showLoginView();
});

/**
 * Check authentication status
 */
async function checkAuth(token) {
    try {
        const response = await fetch('https://journament.com/api/auth/user', {
            headers: {
                'Authorization': `Bearer ${token}`,
                'Accept': 'application/json'
            }
        });

        if (response.ok) {
            const data = await response.json();
            showAuthView(data.user);
            await loadHistory();
        } else {
            showError('Authentication failed. Please log in again.');
            await chrome.storage.sync.remove(['apiToken']);
            showLoginView();
        }
    } catch (error) {
        showError('Failed to connect to Journament. Please check your internet connection.');
        showLoginView();
    }
}

/**
 * Show login view
 */
function showLoginView() {
    document.getElementById('login-view').classList.remove('hidden');
    document.getElementById('auth-view').classList.add('hidden');
}

/**
 * Show authenticated view
 */
function showAuthView(user) {
    document.getElementById('login-view').classList.add('hidden');
    document.getElementById('auth-view').classList.remove('hidden');
    document.getElementById('user-name').textContent = user.name;
    document.getElementById('error-message').classList.add('hidden');
}

/**
 * Show error message
 */
function showError(message) {
    const errorEl = document.getElementById('error-message');
    if (errorEl) {
        errorEl.textContent = message;
        errorEl.classList.remove('hidden');
    }
    // Error element should always exist in popup.html - no fallback needed
}

/**
 * Load import history
 */
async function loadHistory() {
    const { importHistory = [] } = await chrome.storage.local.get(['importHistory']);

    // Update counts
    document.getElementById('import-count').textContent = importHistory.length;

    // Count today's imports
    const today = new Date().toDateString();
    const todayCount = importHistory.filter(item => {
        const itemDate = new Date(item.importedAt).toDateString();
        return itemDate === today;
    }).length;
    document.getElementById('today-count').textContent = todayCount;

    // Display history list
    const historyList = document.getElementById('history-list');
    historyList.innerHTML = '';

    if (importHistory.length === 0) {
        historyList.innerHTML = '<div class="empty-state">No imports yet. Visit an academic article page to start importing!</div>';
        return;
    }

    // Show last 10 imports
    importHistory.slice(0, 10).forEach(item => {
        const div = document.createElement('div');
        div.className = 'history-item';

        const title = item.title || 'Untitled';
        const meta = [item.journal, item.year].filter(Boolean).join(', ');

        div.innerHTML = `
            <div class="history-title" title="${title}">${title}</div>
            <div class="history-meta">${meta || 'No metadata'}</div>
        `;

        if (item.url) {
            div.onclick = () => {
                chrome.tabs.create({ url: item.url });
            };
        }

        historyList.appendChild(div);
    });
}

// Auto-refresh history every 5 seconds if popup is open
setInterval(async () => {
    const { apiToken } = await chrome.storage.sync.get(['apiToken']);
    if (apiToken && !document.getElementById('auth-view').classList.contains('hidden')) {
        await loadHistory();
    }
}, 5000);
