/**
 * Content Script - Runs on every webpage
 * Detects articles and shows import toast
 */

// Load metadata extractor
const extractor = new MetadataExtractor();

// Check if article is detected on page load
window.addEventListener('load', () => {
    detectArticle();
});

// Also check after a short delay (for dynamic content)
setTimeout(() => {
    detectArticle();
}, 2000);

function detectArticle() {
    const metadata = extractor.extract();

    if (metadata) {
        showToastNotification(metadata);
    }
}

function showToastNotification(metadata) {
    // Check if toast already exists
    if (document.getElementById('journament-toast')) {
        return;
    }

    // Create toast element
    const toast = document.createElement('div');
    toast.id = 'journament-toast';
    toast.className = 'journament-toast journament-toast-enter';

    const summary = extractor.getFormattedSummary();

    toast.innerHTML = `
        <div class="journament-toast-header">
            <img src="${chrome.runtime.getURL('icons/icon48.png')}" alt="Journament" class="journament-toast-icon">
            <strong>Article Detected</strong>
            <button class="journament-toast-close" onclick="this.closest('.journament-toast').remove()">×</button>
        </div>
        <div class="journament-toast-body">
            <div class="journament-toast-title">${metadata.title || 'Untitled Article'}</div>
            <div class="journament-toast-meta">${summary}</div>
        </div>
        <div class="journament-toast-actions">
            <button class="journament-btn journament-btn-primary" id="journament-import-btn">
                Import to Journament
            </button>
            <button class="journament-btn journament-btn-secondary" onclick="this.closest('.journament-toast').remove()">
                Dismiss
            </button>
        </div>
    `;

    document.body.appendChild(toast);

    // Add click handler for import button
    document.getElementById('journament-import-btn').addEventListener('click', () => {
        importArticle(metadata, toast);
    });

    // Auto-dismiss after 15 seconds
    setTimeout(() => {
        if (toast.parentElement) {
            toast.classList.add('journament-toast-exit');
            setTimeout(() => toast.remove(), 300);
        }
    }, 15000);
}

async function importArticle(metadata, toast) {
    const button = document.getElementById('journament-import-btn');
    button.disabled = true;
    button.textContent = 'Importing...';

    try {
        // Send message to background script to handle API call
        const response = await chrome.runtime.sendMessage({
            action: 'importArticle',
            metadata: metadata
        });

        if (response.success) {
            showSuccessNotification(response.data);
            toast.remove();
        } else {
            showErrorNotification(response.error);
            button.disabled = false;
            button.textContent = 'Import to Journament';
        }
    } catch (error) {
        console.error('[Journament] Import failed:', error);
        showErrorNotification(error.message);
        button.disabled = false;
        button.textContent = 'Import to Journament';
    }
}

function showSuccessNotification(data) {
    const notification = document.createElement('div');
    notification.className = 'journament-notification journament-notification-success';
    notification.innerHTML = `
        <div class="journament-notification-content">
            <span class="journament-notification-icon">✓</span>
            <div>
                <strong>Article Imported!</strong>
                <p>Successfully added to your Journament library</p>
                ${data.url ? `<a href="${data.url}" target="_blank" class="journament-link">View Article →</a>` : ''}
            </div>
        </div>
    `;
    document.body.appendChild(notification);

    setTimeout(() => notification.remove(), 5000);
}

function showErrorNotification(error) {
    const notification = document.createElement('div');
    notification.className = 'journament-notification journament-notification-error';
    notification.innerHTML = `
        <div class="journament-notification-content">
            <span class="journament-notification-icon">✕</span>
            <div>
                <strong>Import Failed</strong>
                <p>${error || 'An error occurred'}</p>
            </div>
        </div>
    `;
    document.body.appendChild(notification);

    setTimeout(() => notification.remove(), 5000);
}
