/**
 * Background Service Worker
 * Handles API communication with Journament
 */

const API_BASE_URL = 'https://journament.com/api';

// Listen for messages from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'importArticle') {
        handleImportArticle(request.metadata)
            .then(result => sendResponse(result))
            .catch(error => sendResponse({ success: false, error: error.message }));
        return true; // Will respond asynchronously
    }

    if (request.action === 'checkAuth') {
        checkAuthentication()
            .then(result => sendResponse(result))
            .catch(error => sendResponse({ authenticated: false, error: error.message }));
        return true;
    }

    if (request.action === 'saveToken') {
        // Save token from callback page
        chrome.storage.sync.set({ apiToken: request.token })
            .then(() => {
                // Show badge to indicate success
                chrome.action.setBadgeText({ text: '✓' });
                chrome.action.setBadgeBackgroundColor({ color: '#4CAF50' });
                setTimeout(() => chrome.action.setBadgeText({ text: '' }), 3000);

                sendResponse({ success: true });
            })
            .catch(error => {
                console.error('[Journament] Failed to save token:', error);
                sendResponse({ success: false, error: error.message });
            });
        return true;
    }

    if (request.action === 'startOAuth') {
        // Handle OAuth login flow
        handleOAuthLogin(request.provider)
            .then(() => sendResponse({ success: true }))
            .catch(error => sendResponse({ success: false, error: error.message }));
        return true;
    }
});

/**
 * Handle article import
 */
async function handleImportArticle(metadata) {
    // Get API token from storage
    const { apiToken } = await chrome.storage.sync.get(['apiToken']);

    if (!apiToken) {
        throw new Error('Not authenticated. Please log in first.');
    }

    // First, check if article already exists
    const existsResult = await checkArticleExists(metadata);

    if (existsResult.exists) {
        return {
            success: true,
            data: {
                message: 'Article already in database',
                url: `https://journament.com/bibliographies/${existsResult.bibliography_id}`,
                existed: true
            }
        };
    }

    // Import article
    const response = await fetch(`${API_BASE_URL}/articles/import`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${apiToken}`,
            'Accept': 'application/json'
        },
        body: JSON.stringify({
            title: metadata.title,
            authors: metadata.authors,
            abstract: metadata.abstract,
            year: metadata.year,
            venue: metadata.journal,
            doi: metadata.doi,
            source: 'extension',
            url: metadata.url
        })
    });

    if (!response.ok) {
        if (response.status === 401) {
            throw new Error('Authentication expired. Please log in again.');
        }
        const error = await response.json();
        throw new Error(error.message || 'Import failed');
    }

    const data = await response.json();

    // Track import in history
    await saveImportHistory(metadata, data);

    return {
        success: true,
        data: {
            message: 'Article imported successfully',
            url: data.redirect_url || `https://journament.com/bibliographies/${data.bibliography_id}`,
            existed: false
        }
    };
}

/**
 * Check if article already exists
 */
async function checkArticleExists(metadata) {
    let params = new URLSearchParams();

    if (metadata.doi) {
        params.append('doi', metadata.doi);
    } else if (metadata.title) {
        params.append('title', metadata.title);
    } else {
        return { exists: false };
    }

    try {
        const response = await fetch(`${API_BASE_URL}/articles/check?${params}`, {
            headers: {
                'Accept': 'application/json'
            }
        });

        if (response.ok) {
            return await response.json();
        }
    } catch (error) {
    }

    return { exists: false };
}

/**
 * Check authentication status
 */
async function checkAuthentication() {
    const { apiToken } = await chrome.storage.sync.get(['apiToken']);

    if (!apiToken) {
        return { authenticated: false };
    }

    try {
        const response = await fetch(`${API_BASE_URL}/auth/user`, {
            headers: {
                'Authorization': `Bearer ${apiToken}`,
                'Accept': 'application/json'
            }
        });

        if (response.ok) {
            const data = await response.json();
            return {
                authenticated: true,
                user: data.user
            };
        }
    } catch (error) {
        console.error('[Journament] Auth check failed:', error);
    }

    return { authenticated: false };
}

/**
 * Save import to history
 */
async function saveImportHistory(metadata, importData) {
    const { importHistory = [] } = await chrome.storage.local.get(['importHistory']);

    const historyEntry = {
        title: metadata.title,
        authors: metadata.authors,
        journal: metadata.journal,
        year: metadata.year,
        doi: metadata.doi,
        url: importData.redirect_url,
        importedAt: new Date().toISOString()
    };

    // Keep last 50 imports
    importHistory.unshift(historyEntry);
    if (importHistory.length > 50) {
        importHistory.pop();
    }

    await chrome.storage.local.set({ importHistory });
}

/**
 * Handle OAuth login flow
 */
async function handleOAuthLogin(provider) {
    const authUrl = provider === 'google'
        ? 'https://journament.com/auth/google'
        : 'https://journament.com/login/facebook';


    // Create new tab for OAuth
    const tab = await chrome.tabs.create({ url: authUrl });
    const loginTabId = tab.id;

    // Monitor tab URL changes to detect when login is complete
    const listener = (tabId, changeInfo, tab) => {
        if (tabId !== loginTabId) return;

        if (changeInfo.url) {

            // Check if we've reached a post-login page
            const isDashboard = changeInfo.url.includes('/dashboard');
            const isHome = changeInfo.url.includes('/home');
            const isJournament = changeInfo.url.includes('journament.com');
            const isAuthPage = changeInfo.url.includes('/auth/') || changeInfo.url.includes('/login');

            // If we're on journament.com but not on auth/login pages, OAuth is complete
            if (isJournament && !isAuthPage && (isDashboard || isHome || changeInfo.url.match(/journament\.com\/?$/))) {

                // Remove listener
                chrome.tabs.onUpdated.removeListener(listener);

                // Wait a bit for session to be established, then redirect to callback
                setTimeout(() => {
                    chrome.tabs.update(tabId, {
                        url: 'https://journament.com/extension/callback'
                    }).catch(err => {
                        console.error('[Journament] Failed to redirect to callback:', err);
                    });
                }, 1000);
            }
        }
    };

    chrome.tabs.onUpdated.addListener(listener);

    // Clean up listener if tab is closed
    chrome.tabs.onRemoved.addListener(function tabRemovedListener(closedTabId) {
        if (closedTabId === loginTabId) {
            chrome.tabs.onUpdated.removeListener(listener);
            chrome.tabs.onRemoved.removeListener(tabRemovedListener);
        }
    });
}

/**
 * Update badge when articles are detected
 */
chrome.runtime.onMessage.addListener((request, sender) => {
    if (request.action === 'articleDetected' && sender.tab) {
        chrome.action.setBadgeText({
            text: '1',
            tabId: sender.tab.id
        });
        chrome.action.setBadgeBackgroundColor({
            color: '#4CAF50',
            tabId: sender.tab.id
        });
    }
});
